// SPDX-License-Identifier: MIT

#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>

#include "../abos-tools.h"
#include "emmc.h"
#include "mmc.h"

int emmc_setup(const char *dev, struct emmc_state *state)
{
	__u8 ext_csd[512];
	int ret = 0;
	size_t i;
	const struct {
		const char *name;
		__u8 offset;
		__u8 mask;
		__u8 value;
	} ext_csd_vals[] = {
		{ "BKOPS_EN", EXT_CSD_BKOPS_EN, 0xff, BKOPS_MAN_ENABLE },
		{
			"SELF_REFRESH_EN",
			EXT_CSD_SELF_REFRESH_EN,
			EXT_CSD_SELF_REFRESH_FIELD_EN_RTC |
				EXT_CSD_SELF_REFRESH_FIELD_EN,
			EXT_CSD_SELF_REFRESH_FIELD_EN_RTC |
				EXT_CSD_SELF_REFRESH_FIELD_EN,
		},
		{ "SELF_REFRESH_DELAY", EXT_CSD_SELF_REFRESH_DELAY1, 0xff,
		  EXT_CSD_SELF_REFRESH_FIELD_DELAY(
			  6, EXT_CSD_SELF_REFRESH_FIELD_DELAY_FLAG_10SEC) },
		{ "SELF_REFRESH_DELAY2", EXT_CSD_SELF_REFRESH_DELAY2, 0xff,
		  EXT_CSD_SELF_REFRESH_FIELD_DELAY(
			  1, EXT_CSD_SELF_REFRESH_FIELD_DELAY_FLAG_100MS) },
	};

	_cleanup_(closep) int fd = xopen(dev, O_RDWR);

	if (read_extcsd(fd, ext_csd) < 0) {
		ret = -errno;
		fprintf(stderr, "Could not read EXT_CSD from %s: %m\n", dev);
		return ret;
	}

	if (ext_csd[EXT_CSD_REV] < EXT_CSD_REV_V5_1) {
		fprintf(stderr, "%s does not support self refresh function\n",
			dev);
		exit(2);
	}

	for (i = 0; i < sizeof(ext_csd_vals) / sizeof(ext_csd_vals[0]); i++) {
		const char *name = ext_csd_vals[i].name;
		const __u8 offset = ext_csd_vals[i].offset;
		const __u8 mask = ext_csd_vals[i].mask;
		const __u8 val = ext_csd_vals[i].value;

		if (ext_csd[offset] & mask) {
			fprintf(stderr,
				"EXT_CSD[%u] (%s) already programmed: %#x\n",
				offset, name, ext_csd[offset]);
			if ((ext_csd[offset] & mask) != val) {
				fprintf(stderr,
					"Warning: expected EXT_CSD[%u] & %#x = %#x, got %#x\n",
					offset, mask, val,
					ext_csd[offset] & mask);
				ret++;
			}
			continue;
		}
		if (state->verbose)
			printf("writing EXT_CSD[%u] = %#x\n", offset, val);
		if (write_extcsd_value(fd, offset, val, 0) < 0) {
			fprintf(stderr,
				"Could not write EXT_CSD[%u] (%s) = %#x: %m\n",
				offset, name, val);
			ret++;
		}
	}

	if (ret)
		fprintf(stderr,
			"Warning: some values are unexpected values.\n");
	fprintf(stdout, "Done.\n");
	return ret;
}
