// SPDX-License-Identifier: MIT

#include <fcntl.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/time.h>

#include "../abos-tools.h"
#include "emmc.h"
#include "mmc.h"

#define EMMC_CMD_SET_TIME 49
#define EMMC_CMD_GEN_CMD 56
#define SSR_BUFF_SIZE 512 /** Secure Smart Report */
#define GEN_CMD_SSR 0x39 /** GEN_CMD arg for SSR feature */
#define OFFSET_CDATE 88 /** Offset bytes for completion date */

#define SECONDS_IN_A_DAY 86401
/** Offset seconds to convert to Gregorian time from epoch time */
#define EPOCH_OFFSET 62167219200

#define RTC_INFO_TYPE 1 /** payload block type for SET_TIME */

/**
 * Send SET_TIME command.
 *  @param [in] dev       device file path (for error message only)
 *  @param [in] fd        device file fd. Set only eMMC 5.1 or later devices.
 *  @param [in] abs_time  seconds from Jan.1st, year 0 AD till current time in UTC.
 *  @return succeeded(0)/failed(!0)
 */
static int set_time(const char *dev, int fd, __u64 abs_time)
{
	struct mmc_ioc_cmd odata;
	int ret = 0;
	struct {
		__u8 version;
		__u8 info_type;
		__u64 abs_time;
		__u8 reserved[502];
	} __attribute__((packed)) rtc_info_blk = {
		/** regulated in JESD84-B51 */
		.version = 1, /** version structure: shall be set to 1 */
		.reserved = { 0 }, /** shall be set to all zero */
	};

	/* pack mmc_ioc_cmd */
	memset(&odata, 0, sizeof(struct mmc_ioc_cmd));
	odata.write_flag = 1;
	odata.opcode = EMMC_CMD_SET_TIME;
	odata.arg = 0;
	odata.flags = MMC_RSP_SPI_R1 | MMC_RSP_R1 | MMC_CMD_ADTC;
	odata.blksz = sizeof(rtc_info_blk);
	odata.blocks = 1;

	/* pack rtc_info_blk */
	rtc_info_blk.info_type = RTC_INFO_TYPE;
	rtc_info_blk.abs_time = abs_time;
	mmc_ioc_cmd_set_data(odata, &rtc_info_blk);

	ret = ioctl(fd, MMC_IOC_CMD, &odata);
	if (ret) {
		fprintf(stderr, "Failed to ioctl(CMD49) %s %m\n", dev);
		return 1;
	}

	return 0;
}

/**
 * Get last self refresh completion date.
 *  @param [in]  dev       device file path (for error message only)
 *  @param [in]  fd        device file fd. Set only eMMC 5.1 or later devices.
 *  @param [out] abs_time  seconds from Jan.1st, year 0 AD till current time in UTC.
 *  @return succeeded(0)/failed(!0)
 */
static int get_time(const char *dev, int fd, __u64 *abs_time)
{
	struct mmc_ioc_cmd idata;
	int ret = 0;
	__u8 ssr_buf[SSR_BUFF_SIZE] = { 0 };

	/* pack mmc_ioc_cmd */
	memset(&idata, 0, sizeof(struct mmc_ioc_cmd));
	idata.write_flag = 0;
	idata.opcode = EMMC_CMD_GEN_CMD;
	idata.arg = GEN_CMD_SSR;
	idata.flags = MMC_RSP_SPI_R1 | MMC_RSP_R1 | MMC_CMD_ADTC;
	idata.blksz = SSR_BUFF_SIZE;
	idata.blocks = 1;
	mmc_ioc_cmd_set_data(idata, &ssr_buf);

	ret = ioctl(fd, MMC_IOC_CMD, &idata);
	if (ret) {
		fprintf(stderr, "Failed to ioctl(CMD56) %s: %m\n", dev);
		return 1;
	}

	*abs_time = *((__u64 *)&ssr_buf[OFFSET_CDATE]);

	return 0;
}

int emmc_selfrefresh(const char *dev, struct emmc_state *state)
{
	__u64 abs_time = 0;

	_cleanup_(closep) int fd = xopen(dev, O_RDWR);

	if (state->use_system_time) {
		struct timeval now;
		if (gettimeofday(&now, NULL) < 0) {
			fprintf(stderr, "Failed at getimeofday() %m\n");
			return 1;
		}
		/* convert epoch time to Gregorian time */
		abs_time = (__u64)now.tv_sec + EPOCH_OFFSET;

		if (state->verbose)
			printf("system time date: %llu\n", abs_time);
	} else {
		if (get_time(dev, fd, &abs_time))
			return 1;
		if (state->verbose)
			printf("%s last completion date: %llu\n", dev,
			       abs_time);

		if (abs_time == (__u64)UINT64_MAX)
			abs_time = 0;
		abs_time += SECONDS_IN_A_DAY;
	}

	return set_time(dev, fd, abs_time);
}
