// SPDX-License-Identifier: MIT
/* simpler mkimage with minimum options */

#define _GNU_SOURCE
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>
#include <stdio.h>
#include <sys/stat.h>
#include <getopt.h>
#include <stddef.h>
#include <stdbool.h>
#include <stdarg.h>
#include <errno.h>
#include <inttypes.h>
#include <unistd.h>
#include <sys/syscall.h>
#include <zlib.h>

#include "abos-tools.h"

/* from uboot include/image.h */
#define IH_NMLEN 32
/*
 * Legacy format image header,
 * all data in network byte order (aka natural aka bigendian).
 */
struct image_header {
	uint32_t ih_magic; /* Image Header Magic Number    */
	uint32_t ih_hcrc; /* Image Header CRC Checksum    */
	uint32_t ih_time; /* Image Creation Timestamp     */
	uint32_t ih_size; /* Image Data Size              */
	uint32_t ih_load; /* Data  Load  Address          */
	uint32_t ih_ep; /* Entry Point Address          */
	uint32_t ih_dcrc; /* Image Data CRC Checksum      */
	uint8_t ih_os; /* Operating System             */
	uint8_t ih_arch; /* CPU architecture             */
	uint8_t ih_type; /* Image Type                   */
	uint8_t ih_comp; /* Compression Type             */
	uint8_t ih_name[IH_NMLEN]; /* Image Name           */
} __attribute__((packed));

/* data itself is prefixed by an array of u32 sizes terminated by 0
 * we only have one file here
 */
#define DATA_HEADER_SIZE 8

static int print_usage(FILE *out, char *progname)
{
	xfprintf(out, "Usage: %s [-q] script.txt [script.scr]\n", progname);
	xfprintf(out, "Equivalent to mkimage -T script -C none -a 0 -e 0 \\\n");
	xfprintf(
		out,
		"                      -n \"boot script\" -d script.txt bootscript\n");
	xfprintf(out, "\n");
	xfprintf(out, "Options:\n");
	xfprintf(out, "    -q: quiet output\n");
	xfprintf(out, "\n");
	xfprintf(
		out,
		"If output name is omitted, source path with .scr extension is used\n");
	return 0;
}

static ssize_t read_file(const char *path, char **data)
{
	int rc;
	_cleanup_(closep) int fd = xopen(path, O_RDONLY);

	struct stat sb;
	if (fstat(fd, &sb) < 0) {
		rc = -errno;
		perror("fstat source text");
		return rc;
	}

	_cleanup_(freep) char *text =
		xmalloc(sb.st_size + DATA_HEADER_SIZE + 1);

	/* real file size here, return topped up size */
	((uint32_t *)text)[0] = htobe32(sb.st_size);
	((uint32_t *)text)[1] = 0;
	text[sb.st_size + DATA_HEADER_SIZE] = 0;

	/* assume no short reads */
	ssize_t n = read(fd, text + DATA_HEADER_SIZE, sb.st_size);
	if (n < 0) {
		rc = -errno;
		perror("read source text");
		return rc;
	}
	if (n != sb.st_size) {
		fprintf(stderr, "short read for source text\n");
		return -EIO;
	}

	/* Use strlen to check for early nul byte -- it's not forbidden
	 * per se but there should be no valid use for these except getting
	 * arguments wrong
	 */
	size_t len = strlen(text + DATA_HEADER_SIZE);
	if (len != (size_t)sb.st_size) {
		fprintf(stderr,
			"source file %s contained a NUL byte at %zd (of %ld), wrong argument?\n",
			path, len, sb.st_size);
		return -EIO;
	}

	*data = steal_pointer(&text);
	return sb.st_size + DATA_HEADER_SIZE;
}

static ssize_t write_full(int fd, void *data, ssize_t size)
{
	ssize_t n = write(fd, data, size);
	if (n < 0) {
		n = -errno;
		perror("write data");
		return n;
	}
	if (n != size) {
		fprintf(stderr, "short write for script data\n");
		return -EIO;
	}
	return n;
}

int mkbootscr_main(int argc, char *argv[])
{
	uint32_t fake_timestamp = 0;
	bool quiet = false;

	while (true) {
		static const struct option long_options[] = {
			{ "help", no_argument, NULL, 'h' },
			{ "quiet", no_argument, NULL, 'q' },
			{ "timestamp", required_argument, NULL, 'T' },
			{ NULL, 0, NULL, 0 }
		};

		int c = getopt_long(argc, argv, "hT:q", long_options, NULL);
		if (c == -1) {
			break;
		}

		switch (c) {
		case 'h':
			return print_usage(stdout, argv[0]);
		case 'q':
			quiet = true;
			break;
		case 'T':
			fake_timestamp = strtol(optarg, NULL, 0);
			break;
		case '?':
			print_usage(stderr, argv[0]);
			return 1;
		default:
			fprintf(stderr,
				"?? getopt returned character code 0%o ??\n",
				c);
			return 1;
		}
	}

	if (argc - optind > 2 || argc - optind < 1) {
		print_usage(stderr, argv[0]);
		return 1;
	}
	/* we don't care about big images, just read full script in memory */
	char *text_path = argv[optind];
	_cleanup_(freep) char *text = NULL;
	ssize_t text_size = read_file(text_path, &text);
	if (text_size < 0)
		return 1;
	_cleanup_(unlinkp) char *out_path = NULL;

	if (argc - optind == 2) {
		out_path = xstrdup(argv[optind + 1]);
	} else {
		char *dot = strrchr(text_path, '.');
		char *slash = strrchr(text_path, '/');
		if (!dot || (slash && dot < slash)) {
			dot = text_path + strlen(text_path);
		}
		size_t len = dot - text_path;
		out_path = xmalloc(len + 5 /* .scr\0 */);
		strncpy(out_path, text_path, len);
		strcpy(out_path + len, ".scr");
	}
	_cleanup_(closep) int script_fd =
		xopen(out_path, O_WRONLY | O_TRUNC | O_CREAT, 0666);

	/* actual mkimage */
	struct image_header header = { 0 };
	header.ih_magic = 0x56190527;
	header.ih_time = htobe32(fake_timestamp ?: time(NULL));
	header.ih_size = htobe32(text_size);
	header.ih_dcrc = htobe32(crc32(0, (uint8_t *)text, text_size));
	header.ih_type = 6; /* IH_TYPE_SCRIPT */
	strcpy((char *)header.ih_name, "boot script");

	/* must be last with crc previously zeroed out */
	header.ih_hcrc = htobe32(crc32(0, (uint8_t *)&header, sizeof(header)));
	if (write_full(script_fd, &header, sizeof(header)) < 0)
		return 1;
	if (write_full(script_fd, text, text_size) < 0)
		return 1;

	if (!quiet)
		xfprintf(stdout, "Wrote %s\n", out_path);

	/* don't unlink file at exit */
	out_path[0] = 0;

	return EXIT_SUCCESS;
}
