// SPDX-License-Identifier: MIT
#define _XOPEN_SOURCE 700
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <errno.h>
#include <stdbool.h>
#include <linux/fs.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <fcntl.h>

#include "abos-tools.h"

static void print_usage(FILE *stream, const char *argv0)
{
	fprintf(stream, "Usage: %s [-f|-u] directory\n", argv0);
	fprintf(stream, "\n");
	fprintf(stream, "  -f, --freeze: freeze filesystem\n");
	fprintf(stream, "  -u, --unfreeze: unfreeze filesystem\n");
	fprintf(stream, "  -h, --help: print this help\n");
	fprintf(stream, "  -V, --version: print version\n");
}

int fsfreeze_main(int argc, char *argv[])
{
	unsigned long op = 0;
	while (true) {
		static const struct option long_options[] = {
			{ "freeze", no_argument, NULL, 'f' },
			{ "unfreeze", no_argument, NULL, 'u' },
			{ "help", no_argument, NULL, 'h' },
			{ "version", no_argument, NULL, 'V' },
			{ NULL, 0, NULL, 0 }
		};
		int c;

		c = getopt_long(argc, argv, "fuhV", long_options, NULL);
		if (c == -1) {
			break;
		}

		switch (c) {
		case 'f':
			op = FIFREEZE;
			break;
		case 'u':
			op = FITHAW;
			break;
		case 'h':
			print_usage(stdout, argv[0]);
			return EXIT_SUCCESS;
		case 'V':
			printf("abos-tool fsfreeze version 0.1\n");
			return EXIT_SUCCESS;
		case '?':
			print_usage(stderr, argv[0]);
			return EXIT_FAILURE;
		default:
			fprintf(stderr,
				"?? getopt returned character code 0%o ??\n",
				c);
			return EXIT_FAILURE;
		}
	}

	if (argc - optind != 1) {
		// this is a shame as we could just go through each fs we want to freeze,
		// but that's compatibility for you...
		fprintf(stderr, "Must have exactly one directory argument\n");
		return EXIT_FAILURE;
	}
	if (op == 0) {
		fprintf(stderr, "Either --freeze or --unfreeze must be set\n");
		return EXIT_FAILURE;
	}

	const char *dir = argv[optind];
	_cleanup_(closep) int fd = open(dir, O_RDONLY | O_DIRECTORY);
	if (fd < 0) {
		fprintf(stderr, "Could not open '%s': %s\n", dir,
			strerror(errno));
		return EXIT_FAILURE;
	}
	if (ioctl(fd, op) < 0) {
		fprintf(stderr, "Could not %s '%s': %s\n",
			op == FIFREEZE ? "freeze" : "unfreeze", dir,
			strerror(errno));
		return EXIT_FAILURE;
	}
	return EXIT_SUCCESS;
}
