# SPDX-License-Identifier: MIT

certificates_help() {
	echo "Usage: abos-ctrl certificates [action] [... more actions]"
	echo
	echo "Possible actions:"
	echo "  list: list currently allowed certificates"
	echo "  reset: equivalent to restore-onetime restore-atmark remove-user,"
	echo "         plus removes initial_setup version to allow re-installing it"
	echo "  restore-onetime: re-add one-time public certificate"
	echo "  remove-onetime: remove one-time public certificate"
	echo "  restore-atmark: re-add atmark certificates (removes old ones if present)"
	echo "  remove-atmark: remove atmark certificates"
	echo "  remove-user: remove user-provided certificates"
	echo "  restore-atmark-old: re-add old atmark certificates"
}

certs_list() {
	[ -e /etc/swupdate.pem ] || error "No allowed certificates!"

	# After installing anyhting on mkswu >= 4.11 each cert will always
	# have comments, so we can just list comments as follow.
	# Just error if that is not the case to avoid not listing a cert.
	if ! awk '/^#/ { comment=1 }
		  /----BEGIN/ && !comment { exit(1) }
		  /----BEGIN/ { comment=0 }
	    ' /etc/swupdate.pem; then
		error "/etc/swupdate.pem might be malformed, refusing to list"
	fi

	sed -ne 's/^# /- /p' /etc/swupdate.pem
}

certs_has_onetime() {
	local crt onetime

	# We keep the onetime-public certificate first in file to keep this check simple
	crt="$(openssl x509 -noout -in /etc/swupdate.pem -pubkey  2>/dev/null \
                | sed -e '/-----/d' | tr -d '\n')"
        onetime="MFYwEAYHKoZIzj0CAQYFK4EEAAoDQgAEYTN7NghmISesYQ1dnby5YkocLAe2/EJ8OTXkx/xGhBVlJ57eGOovtPORd/JMkA6lWI0N/pD5p6eUGcwrQvRtsw=="

	[ "$crt" = "$onetime" ]
}

certs_restore_onetime() {
	if [ -n "$onetime_seen" ]; then
		info "one-time public certificate already present"
		return
	fi
	modified=1

	info "Restoring swupdate-onetime-public.pem"
	# keep onetime cert first to keep certs_has_onetime() simple,
	# use index 0 (normally starts at 1)
	cat > "$tmpdir/cert.0.onetime" << EOF \
		|| error "Could not write onetime public key"
# swupdate-onetime-public.pem
-----BEGIN CERTIFICATE-----
MIIB2jCCAYCgAwIBAgIUJq1g2JPwnHdeezaY4myfJCj1SJgwCgYIKoZIzj0EAwIw
RDERMA8GA1UECgwIU1dVcGRhdGUxLzAtBgNVBAMMJkFybWFkaWxsbyBzd3VwZGF0
ZSBwdWJsaWMgb25lLXRpbWUga2V5MB4XDTIxMTAwODAyMDEzM1oXDTI2MTAwNzAy
MDEzM1owRDERMA8GA1UECgwIU1dVcGRhdGUxLzAtBgNVBAMMJkFybWFkaWxsbyBz
d3VwZGF0ZSBwdWJsaWMgb25lLXRpbWUga2V5MFYwEAYHKoZIzj0CAQYFK4EEAAoD
QgAEYTN7NghmISesYQ1dnby5YkocLAe2/EJ8OTXkx/xGhBVlJ57eGOovtPORd/JM
kA6lWI0N/pD5p6eUGcwrQvRts6NTMFEwHQYDVR0OBBYEFJH/mOi6ZmaQn/o1qIuI
U/937NB2MB8GA1UdIwQYMBaAFJH/mOi6ZmaQn/o1qIuIU/937NB2MA8GA1UdEwEB
/wQFMAMBAf8wCgYIKoZIzj0EAwIDSAAwRQIhAOLfV5A4z5LGmN3wDpeCYkhPck2t
nK0fTAq/GXsI1Um1AiB0TkLYr/aNh1fkdjLbmZSV9YYzB1OQPF4R957CMBxusA==
-----END CERTIFICATE-----
EOF
	onetime_seen=1
}

certs_remove_onetime() {
	if [ -z "$onetime_seen" ]; then
		return
	fi
	info "Removing swupdate-onetime-public.pem"
	modified=1
	rm -f  "$tmpdir"/cert.*.onetime || error "Could not remove onetime cert"
	onetime_seen=""
}

certs_restore_atmark() {
	local atmark_ref_count

	atmark_ref_count=$(grep -c "BEGIN CERTIFICATE" "$scripts_dir/certificates_atmark.pem") \
		|| error "Missing certificates_atmark.pem ?"
	if [ "$atmark_count" = "$atmark_ref_count" ] && [ "$atmark_old_count" = 0 ]; then
		info "Atmark certificates already present"
		return
	fi
	modified=1

	rm -f "$tmpdir"/cert.*.atmark "$tmpdir"/cert.*.atmark_old \
		|| error "Could not remove temporary file"

	# These files normally contain a single cert, but it
	# does not matter if there are more: just copy it all.
	last_idx=$((last_idx+1))
	cp "$scripts_dir/certificates_atmark.pem" "$tmpdir/cert.$last_idx.atmark" \
		|| error "Could not write temporary file"
	atmark_count="$atmark_ref_count"
	atmark_old_count=0
}

certs_restore_atmark_old() {
	local atmark_ref_count

	atmark_ref_count=$(grep -c "BEGIN CERTIFICATE" "$scripts_dir/certificates_atmark_old.pem") \
		|| error "Missing certificates_atmark_old.pem ?"
	if [ "$atmark_old_count" = "$atmark_ref_count" ]; then
		info "Atmark old certificates already present"
		return
	fi
	modified=1

	rm -f "$tmpdir"/cert.*.atmark_old \
		|| error "Could not remove temporary file"

	# These files normally contain a single cert, but it
	# does not matter if there are more: just copy it all.
	last_idx=$((last_idx+1))
	cp "$scripts_dir/certificates_atmark_old.pem" "$tmpdir/cert.$last_idx.atmark" \
		|| error "Could not write temporary file"
	atmark_old_count="$atmark_ref_count"
}

certs_remove_atmark() {
	if [ "$((atmark_count + atmark_old_count))" = 0 ]; then
	       return
	fi
	modified=1

	rm -f "$tmpdir"/cert.*.atmark "$tmpdir"/cert.*.atmark_old \
		|| error "Could not remove temporary file"
	atmark_count=0
	atmark_old_count=0
}

certs_remove_user() {
	local cert comment

	for cert in "$tmpdir"/cert.*; do
		case "$cert" in
		*.onetime|*.atmark)
			# skip atmark certs
			;;
		*)
			comment=$(head -n 1 "$cert" 2>/dev/null)
			if [ "${comment#\# }" != "$comment" ]; then
				info "Removing \"${comment#\# }\""
			else
				info "Removing $cert"
			fi
			modified=1
			rm -f "$cert" || error "Could not remove $cert"
			;;
		esac
	done
}


certs_reset() {
	clear_initial_setup=1

	certs_restore_onetime
	certs_restore_atmark
	certs_remove_user
}

certs_explode_swupdate_pem() {
	# explodes swupdate.pem in target directory and print last used index
	if ! [ -e /etc/swupdate.pem ]; then
		# warning goes to stderr
		warning "no /etc/swupdate.pem - there might be a problem"
		# no index used
		echo 0
		return
	fi

	awk -v certsdir="$tmpdir" '
                ! outfile { idx++; outfile=certsdir "/cert." idx }
             outfile { print > outfile }
             /END CERTIFICATE/ { outfile="" }
	     END { print idx }
             ' /etc/swupdate.pem
}

certs_get_atmark() {
	local cert pubkey

	for cert in "$tmpdir"/cert.*; do
		[ -e "$cert" ] || continue
		pubkey=$(openssl x509 -noout -in "$cert" -pubkey 2>/dev/null | sed -e '/-----/d' | tr -d '\n')
		case "$pubkey" in
		# Armadillo public one-time cert
		"MFYwEAYHKoZIzj0CAQYFK4EEAAoDQgAEYTN7NghmISesYQ1dnby5YkocLAe2/EJ8OTXkx/xGhBVlJ57eGOovtPORd/JMkA6lWI0N/pD5p6eUGcwrQvRtsw==")
			mv "$cert" "$cert.onetime" \
				|| error "Could not rename temporary file"
			onetime_seen=1
			;;
		# atmark-2|atmark-3
		"MFYwEAYHKoZIzj0CAQYFK4EEAAoDQgAERkRP5eTXBTG760gEmBfCBz4fWyYfUx3a+sYyHe4uc1sQN2bavxfaBlJmyGI4MY/Pkjh5FDVcddZfil552WUoWQ=="|\
		"MFYwEAYHKoZIzj0CAQYFK4EEAAoDQgAE6IZHb+5RM8wxXWB8NdVpy5k7THY61SKP7+4GqegW2SDJ3yYUYuwL7MZVjKtauUYUYQVvKzEc+ghxOdQgModzfA==")
			mv "$cert" "$cert.atmark" \
				|| error "Could not rename temporary file"
			atmark_count=$((atmark_count+1))
			;;
		# atmark-1
		"MFYwEAYHKoZIzj0CAQYFK4EEAAoDQgAEjgbd3SI8+iof3TLL9qTGNlQN84VqkESPZ3TSUkYUgTiEL3Bi1QoYzGWGqfdmrLiNsgJX4QA3gpaC19Q+fWOkEA==")
			mv "$cert" "$cert.atmark_old" \
				|| error "Could not rename temporary file"
			atmark_old_count=$((atmark_old_count+1))
			;;
		"")
			# just spaces? delete
			if [ -z "$(cat "$cert")" ]; then
				rm -f "$cert" \
					|| error "Could not remove temporary file"
			else
				warning "This certificate seems corrupted, consider removing this block from /etc/swupdate.pem:"
				warning "$(cat "$cert")"
			fi
			;;
		esac
	done
}

certs_cleanup() {
	if [ -n "$tmpdir" ]; then
		rm -rf "$tmpdir"
	fi
	if [ -n "$tmpfile" ]; then
		rm -f "$tmpfile"
	fi
}

ctrl_certificates() {
	local no_action=1 modified="" clear_initial_setup=""
	local onetime_seen="" atmark_count=0
	local last_idx=""
	# tmpdir/tmpfile are not local because used in trap
	tmpdir="" tmpfile=""

	trap certs_cleanup EXIT INT QUIT TERM
	tmpdir=$(mktemp -d "${TMPDIR:-/tmp}/abos_ctrl_certificates.XXXXXX") \
		|| error "Could not create temporary dir"

	last_idx=$(certs_explode_swupdate_pem) \
		|| error "Could not split swupdate.pem"
	certs_get_atmark

	while [ "$#" -gt 0 ]; do
		case "$1" in
		"restore-onetime")
			certs_restore_onetime
			no_action=
			;;
		"remove-onetime")
			certs_remove_onetime
			no_action=
			;;
		"restore-atmark")
			certs_restore_atmark
			no_action=
			;;
		"restore-atmark-old")
			certs_restore_atmark_old
			no_action=
			;;
		"remove-atmark")
			certs_remove_atmark
			no_action=
			;;
		"remove-user")
			certs_remove_user
			no_action=
			;;
		"reset")
			certs_reset
			no_action=
			;;
		"list")
			[ -n "$no_action" ] \
				|| error "list cannot be requested with other actions"
			certs_list
			return
			;;
		"-q"|"--quiet")
			debug=$((debug-1))
			;;
		"-v"|"--verbose")
			debug=$((debug+1))
			;;
		-[vq]*)
			set -- "$@" "${1:0:2}" "-${1:2}"
			;;
		*)
			warning "Invalid option $1"
			certificates_help >&2
			exit 1
			;;
		esac
		shift
	done

	if [ -n "$no_action" ]; then
		warning "No action requested, falling back to list"
		certs_list
		return
	fi
	if [ -z "$modified" ]; then
		info "swupdate.pem unchanged"
		return
	fi

	if [ -n "$clear_initial_setup" ]; then
		if grep -q '^extra_os\.initial_setup ' /etc/sw-versions; then
			info "Removing 'initial_setup' from /etc/sw-versions"
			sed -i -e '/^extra_os\.initial_setup /d' /etc/sw-versions \
				|| error "Could not remove initial_setup from sw-versions"
		else
			clear_initial_setup=""
		fi
	fi

	tmpfile=$(mktemp /etc/swupdate.pem.XXXXXX) \
		|| error "Could not create temporary swupdate.pem"
	if ! stat "$tmpdir"/cert.* >/dev/null 2>&1; then
		warning "No certificate left, updates will not be possible"
	else
		cat "$tmpdir"/cert.*
	fi > "$tmpfile" \
		&& mv "$tmpfile" /etc/swupdate.pem \
		|| error "Could not update swupdate.pem"
	tmpfile=""

	persist_file /etc/swupdate.pem ${clear_initial_setup:+/etc/sw-versions} \
		|| error "Could not persist swupdate.pem. Warning: it has been udpated in tmpfs"
}
