// SPDX-License-Identifier: MIT
#define _POSIX_C_SOURCE 200809L

#include <ctype.h>
#include <fcntl.h>
#include <libgen.h>
#include <linux/fs.h>
#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

#include "../abos-tools.h"
#include "emmc.h"
#include "mmc.h"

int read_extcsd(int fd, __u8 *ext_csd)
{
	struct mmc_ioc_cmd idata;
	memset(&idata, 0, sizeof(idata));
	memset(ext_csd, 0, sizeof(__u8) * 512);
	idata.write_flag = 0;
	idata.opcode = MMC_SEND_EXT_CSD;
	idata.arg = 0;
	idata.flags = MMC_RSP_SPI_R1 | MMC_RSP_R1 | MMC_CMD_ADTC;
	idata.blksz = 512;
	idata.blocks = 1;
	mmc_ioc_cmd_set_data(idata, ext_csd);

	return ioctl(fd, MMC_IOC_CMD, &idata);
}

int write_extcsd_value(int fd, __u8 index, __u8 value, unsigned int timeout_ms)
{
	struct mmc_ioc_cmd idata;

	memset(&idata, 0, sizeof(idata));
	idata.write_flag = 1;
	idata.opcode = MMC_SWITCH;
	idata.arg = (MMC_SWITCH_MODE_WRITE_BYTE << 24) | (index << 16) |
		    (value << 8) | EXT_CSD_CMD_SET_NORMAL;
	idata.flags = MMC_RSP_SPI_R1B | MMC_RSP_R1B | MMC_CMD_AC;
	/* Kernel will set cmd_timeout_ms if 0 is set */
	idata.cmd_timeout_ms = timeout_ms;

	return ioctl(fd, MMC_IOC_CMD, &idata);
}

/**
 * Read mmc block device attribute from sysfs.
 *  @param [in]  mmc_num  device number to read attribute.
 *  @param [in]  attr_name attribute name of mmc block device.
 *  @param [out]  attr  attribute string of mmc block device
 *  @return succeeded(0)/failed(!0)
 */
static int read_attr(int mmc_num, const char *attr_name, char *attr, int size)
{
	char path[PATH_MAX];

	/**
	 * note: support only when RCA is 0x0001 (linux initial value).
	 */
	int rc = snprintf(path, PATH_MAX,
			  "/sys/class/mmc_host/mmc%d/mmc%d:0001/%s", mmc_num,
			  mmc_num, attr_name);
	xassert(rc >= 0 && rc < PATH_MAX);

	_cleanup_(closep) int fd = open(path, O_RDONLY);
	if (fd < 0)
		return -errno;

	errno = 0;
	rc = read(fd, attr, size);
	if (rc <= 0 || rc >= size)
		return errno ? errno : -ERANGE;

	attr[rc] = 0;
	while (isspace(attr[rc - 1])) {
		rc--;
		attr[rc] = 0;
	}
	return rc;
}

/**
 * Check device whether to support Micron specific commands.
 *  @param [in]  dev device file path. Set only eMMC 5.1 or later devices.
 *  @return Supported(true)/not Supported(false)
 */
bool mmc_is_micron(const char *dev, int verbose)
{
	_cleanup_(freep) char *dup_dev = xstrdup(dev);

	char *mmcblk_name = basename(dup_dev);
	if (mmcblk_name == NULL) {
		fprintf(stderr, "Failed to basename(%s)\n", dup_dev);
		return false;
	}

	int mmc_num;
	if (sscanf(mmcblk_name, "mmcblk%d", &mmc_num) != 1) {
		fprintf(stderr, "%s is not of form mmcblk[0-9]\n", mmcblk_name);
		return false;
	}

	/** device type */
	char type[10];
	if (read_attr(mmc_num, "type", type, sizeof(type)) < 0)
		return false;
	if (strcmp(type, "MMC"))
		return false;

	/** manufacturer ID */
	char _manfid[10];
	if (read_attr(mmc_num, "manfid", _manfid, sizeof(_manfid)) < 0)
		return false;

	errno = 0;
	long manfid = strtol(_manfid, NULL, 16);
	if (errno) {
		fprintf(stderr, "Failed to strtol %s (manfid): %m\n", _manfid);
		return false;
	}

	/** eMMC revision */
	char _rev[10];
	if (read_attr(mmc_num, "rev", _rev, sizeof(_rev)) < 0)
		return false;
	errno = 0;
	long rev = strtol(_rev, NULL, 16);
	if (errno) {
		fprintf(stderr, "Failed to strtol %s (rev): %m\n", _rev);
		return false;
	}

	if (verbose >= 2) {
		printf("device attrs:\n");
		printf("    device   mmc%d\n", mmc_num);
		printf("    type     %s\n", type);
		printf("    manf id  %lx\n", manfid);
		printf("    revision %lx\n", rev);
	}

	return manfid == 0x13 /* micron */ && rev >= 0x08 /* eMMC v5.1 */;
}
