// SPDX-License-Identifier: MIT

#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>

#include "emmc.h"

static void print_usage(FILE *stream, const char *argv0)
{
	fprintf(stream, "Usage: %s [options] mmc\n", argv0);
	fprintf(stream,
		"  Initiate self refresh feature for Micron's eMMC device.\n");
	fprintf(stream, "  Options:\n");
	fprintf(stream, "  -t --time    Use system time resource.\n");
	fprintf(stream, "  -v --verbose Verbose message\n");
	fprintf(stream, "  -V --version Display version number of emmc-sref\n");
	fprintf(stream,
		"  --setup      Run one-time emmc setup instead of setting time\n");
	fprintf(stream, "  -h --help    Display help message.\n");
}

int emmc_sref_main(int argc, char *argv[])
{
	struct emmc_state state = { 0 };
	tune_fn tune_fn = emmc_selfrefresh;
	int rc = 0;

	while (true) {
#define OPT_SETUP 257
		static const struct option long_options[] = {
			{ "time", no_argument, 0, 't' },
			{ "help", no_argument, 0, 'h' },
			{ "verbose", no_argument, 0, 'v' },
			{ "version", no_argument, 0, 'V' },
			{ "setup", no_argument, 0, OPT_SETUP },
			{ NULL, 0, NULL, 0 }
		};
		int c;

		c = getopt_long(argc, argv, "htvV", long_options, NULL);
		if (c == -1) {
			break;
		}

		switch (c) {
		case 't':
			state.use_system_time = true;
			break;
		case OPT_SETUP:
			tune_fn = emmc_setup;
			break;
		case 'v':
			state.verbose++;
			break;
		case 'h':
			print_usage(stdout, argv[0]);
			return EXIT_SUCCESS;
		case 'V':
			printf("abos-tool emmc-sref version 2.0.0\n");
			return EXIT_SUCCESS;
		case '?':
			print_usage(stderr, argv[0]);
			return EXIT_FAILURE;
		default:
			fprintf(stderr,
				"?? getopt returned character code 0%o ??\n",
				c);
		}
	}

	if (optind == argc) {
		fprintf(stderr, "No device given\n");
		return EXIT_FAILURE;
	}

	for (int i = optind; i < argc; i++) {
		const char *dev = argv[i];
		if (!mmc_is_micron(dev, state.verbose)) {
			if (state.verbose)
				printf("Skipping %s\n", dev);
			continue;
		}
		rc |= tune_fn(argv[i], &state);
	}

	return rc;
}
