// SPDX-License-Identifier: MIT
#define _XOPEN_SOURCE 500
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "abos-tools.h"

const struct {
	const char *name;
	int (*main)(int argc, char *argv[]);
	const char *install_path;
} progs[] = { { "fsfreeze", fsfreeze_main, "/sbin/fsfreeze" },
	      { "mkbootscr", mkbootscr_main, "/usr/sbin/mkbootscr" },
	      { "renameat2", renameat2_main, "/usr/bin/renameat2" },
	      { "schedule_ts", schedule_ts_main, "/usr/bin/schedule_ts" },
	      { "emmc-sref", emmc_sref_main, "/usr/sbin/emmc-sref" },
	      { 0 } };

void help(FILE *stream)
{
	int i;

	fprintf(stream, "Usage: abos-tools command [args]\n");
	fprintf(stream, "Usage: abos-tools --install\n");
	fprintf(stream, "Usage: abos-tools --list-all\n");
	fprintf(stream, "\n");
	fprintf(stream, "Available commands:\n");
	for (i = 0; progs[i].name; i++) {
		fprintf(stream, "    %s\n", progs[i].name);
	}
}

int install(const char *target)
{
	int rc = 0;
	int i;

	if (target[0] != '/') {
		fprintf(stderr, "'%s' is not an absolute path\n", target);
		return 2;
	}
	for (i = 0; progs[i].name; i++) {
		if (symlink(target, progs[i].install_path) < 0) {
			if (errno == EEXIST)
				continue;
			fprintf(stderr, "Failed to create '%s': %s\n",
				progs[i].install_path, strerror(errno));
			rc = 3;
		}
	}

	return rc;
}

int list_all(void)
{
	int i;
	for (i = 0; progs[i].name; i++) {
		printf("%s\n", progs[i].install_path);
	}
	return 0;
}

int main(int argc, char *argv[])
{
	const char *prog = NULL;
	int i;

	if (argc < 1)
		return 1;

	prog = strrchr(argv[0], '/');
	if (prog)
		prog++;
	else
		prog = argv[0];

	if (strcmp(prog, "abos-tools") == 0) {
		if (argc < 2) {
			return 1;
		}
		if (strcmp(argv[1], "-h") == 0 ||
		    strcmp(argv[1], "--help") == 0) {
			help(stdout);
			return 0;
		}
		if (strcmp(argv[1], "--install") == 0) {
			return install(argv[0]);
		}
		if (strcmp(argv[1], "--list-all") == 0) {
			return list_all();
		}

		argc--;
		argv++;
		prog = argv[0];
	}
	for (i = 0; progs[i].name; i++) {
		if (strcmp(prog, progs[i].name) == 0)
			return progs[i].main(argc, argv);
	}

	fprintf(stderr, "No such program: %s\n", prog);
	help(stderr);
	return 1;
}
