# SPDX-License-Identifier: MIT

# for certs_check_onetime
. "$scripts_dir/certificates.sh"

status_print_last_update() {
	local otherpart old_versions new_versions versions

	# don't bother if not going to print
	if [ "${debug:-3}" -lt 2 ]; then
		return
	fi

	otherpart=${rootpart##*/}
	otherpart="${otherpart%[12]}$((3-${otherpart##*mmcblk?p}))"
	new_versions="/etc/sw-versions"
	old_versions="/var/log/swupdate/sw-versions-$otherpart"
	if ! [ -e "$old_versions" ]; then
		# last update was made with an old mkswu
		return
	fi

	# from mkswu
	if ! versions="$(awk '
		newvers == 0 { oldv[$1]=$2 }
		newvers == 1 { newv[$1]=$2 }
		END {
			for (comp in newv) {
				old = oldv[comp] != "" ? oldv[comp] : "unset";
				if (old != newv[comp])
					printf("  %s: %s -> %s\n", comp, old, newv[comp]);
			}
			for (comp in oldv) {
				if (newv[comp] == "")
					printf("  %s: %s -> unset\n", comp, oldv[comp]);
			}
		}' "$old_versions" newvers=1 "$new_versions")"; then
			warning "Could not compare new/old versions"
			return
	fi

	info "Last update on $(date -d "@$last_update_ts"), updated:
$versions"
}

status_check_rollback_upgradeavailable() {
	local envdev
	envdev="$(awk '/\/dev/ { print $1; exit; }' /etc/fw_env.config 2>/dev/null)"
	if [ -z "$envdev" ] || ! [ -r "$envdev" ]; then
		# can't read env (no env or non-root user)
		 trace "rollback-status: $status"
		 return
	fi

	case "$(fw_printenv_nowarn upgrade_available)" in
	*upgrade_available=2)
		trace "rollback-status: $status: auto-rollback enabled (cloned)"
		;;
	*upgrade_available=1)
		# this can be visible after e.g. persist_file after rollback-clone
		trace "rollback-status: $status: auto-rollback enabled (post-update)"
		;;
	*upgrade_available=0)
		trace "rollback-status: $status: available, no auto-rollback"
		;;
	*upgrade_available=|*upgrade_available=00)
		trace "rollback-status: $status: no fallback"
		;;
	*)
		warning "upgrade_available uboot environment unreadable or unexpected value"
		trace "rollback-status: $status: ???"
		;;
	esac
}

status_check_rollback() {
	# compare current partition with /var/log/swupdate/last_update
	local last_update_rootpart last_update_ts
	local status="OK"

	[ -z "$check_all$check_rollback" ] && return

	last_update_rootpart="$(cat /var/log/swupdate/last_update 2>/dev/null)"
	last_update_ts="${last_update_rootpart##* }"
	last_update_rootpart="${last_update_rootpart% *}"

	if [ -z "$last_update_rootpart" ]; then
		trace "rollback-status: OK: no fallback (fresh install)"
		return
	fi

	# rolled back, return early
	if [ "$last_update_rootpart" != "$rootpart" ]; then
		warning "Currently running on non-latest version (expected $last_update_rootpart${last_update_ts:+ installed on $(date -d "@$last_update_ts")})"
		status="rolled back"
	else
		status_print_last_update
	fi

	status_check_rollback_upgradeavailable

	# return error if status was not OK
	[ "$status" = "OK" ]
}

status_check_onetime_cert() {
	[ -z "$check_all$check_onetime_cert" ] && return

	if certs_has_onetime; then
		warning "swupdate onetime public certificate is present, anyone can access this device" \
			"Please install initial_setup.swu (from mkswu --init)," \
			"or remove it with 'abos-ctrl certificates remove-onetime'"
		return 1
	fi
}

status_check_modules_installed() {
	[ -z "$check_all" ] && return

	# just check if modules dir exists
	test -d "/lib/modules/$(uname -r)" && return

	# if not, verify we have any module compiled in if we can.
	# absence of /proc/config.gz or presence of 'as module' in it
	# are considered failures.
	if ! [ -e /proc/config.gz ] || zcat /proc/config.gz | grep -q "=m"; then
		warning "This kernel requires modules, but modules were not found in /lib/modules/$(uname -r)." \
			"Please check your kernel installation or re-install default kernel with" \
			"'abos-ctrl restore-kernel'"
		return 1
	fi
}

status_help() {
	echo "Usage: abos-ctrl [common options] status [extra options]"
	echo
	echo "  --onetime-cert    only check for onetime certificate presence"
	echo "  --rollback        only check for rollback status"
	echo "both options imply quiet, use with -v to print warnings"
	echo "or -vvv for maximum messages"
	echo
	echo "see abos-ctrl --help for common options"
}

ctrl_status() {
	local fail=""
	local check_onetime_cert="" check_rollback="" check_all=1
	# status without options prints trace by default
	debug=3

	dev_from_current

	while [ "$#" -gt 0 ]; do
		case "$1" in
		"--onetime-cert")
			check_onetime_cert=1
			if [ -n "$check_all" ]; then
				# no message by default
				debug=$((debug-3))
				check_all=""
			fi
			;;
		"--rollback")
			check_rollback=1
			if [ -n "$check_all" ]; then
				debug=$((debug-3))
				check_all=""
			fi
			;;
		"-q"|"--quiet")
			debug=$((debug-1))
			;;
		"-v"|"--verbose")
			debug=$((debug+1))
			;;
		-[vq]*)
			set -- "$@" "${1:0:2}" "-${1:2}"
			;;
		*)
			warning "Invalid option $1"
			status_help >&2
			exit 1
			;;
		esac
		shift
	done


	trace "Currently booted on $rootpart"

	status_check_rollback || fail=1
	status_check_onetime_cert || fail=1
	status_check_modules_installed || fail=1

	[ -z "$fail" ] || exit 1
}
