# SPDX-License-Identifier: MIT

rollback_clone_help() {
	echo "Usage: abos-ctrl rollback-clone [--fstype <btrfs|ext4>]"
	echo
	echo "Copy current partition to other side for rollback"
	echo "If fstype is passed, it overrides the detection of using the same type"
}

rollback_clone_cleanup() {
	trap - EXIT INT QUIT TERM
	umount_if_mountpoint /target
	[ -n "$target_ab" ] && [ -e "/dev/mapper/rootfs_$target_ab" ] \
		&& cryptsetup luksClose "rootfs_$target_ab"
	unlock_update
}

rollback_clone_copy_rootfs() {
	local plain_dev="$dev"
	# check if we really need to copy first
	luks_unlock "rootfs_$target_ab"

	if opts=,ro mount_rootfs "$dev" /target \
	    && ! [ -e /target/.created ] \
	    && cmp -s /etc/.rootfs_update_timestamp \
			/target/etc/.rootfs_update_timestamp; then
		# already up to date, use as is
		info "Reusing up-to-date rootfs"
		return
	fi
	if is_mountpoint /target; then
		umount /target || error "Could not umount /target we just mounted"
	fi
	if [ -e "/dev/mapper/rootfs_$target_ab" ]; then
		cryptsetup luksClose "rootfs_$target_ab" \
			|| error "Could not deactivate luks volume we just created"
	fi

	info "Cloning rootfs"
	cloned_rootfs=1
	# The standard for encrypting isn't if B side was encrypted, it's if
	# we're encrypted now.
	dev="$plain_dev"
	if cryptsetup isLuks "$rootpart" 2>/dev/null; then
		luks_format "rootfs_$target_ab"
	fi
	# We do not use dd for copy as that can cause problem with
	# duplicate fs uuid/labels. Just mount and copy.
	# cf. mkswu/scripts/pre_rootfs.sh
	[ -e "/boot/extlinux.conf" ] && extlinux=1
	[ -n "$fstype" ] || fstype="$(findmnt -nr -o FSTYPE /live/rootfs 2>/dev/null)"
	case "$fstype" in
	btrfs)
		mkfs.btrfs -q -L "rootfs_${target_ab}" -m dup -f "$dev" \
			|| error "Could not reformat $dev"
		;;
	ext4)
		dd if=/dev/zero of="$dev" bs=32k count=1 status=none
		mkfs.ext4 -q ${extlinux:+-O "^64bit"} -L "rootfs_${target_ab}" -F "$dev" \
			|| error "Could not reformat $dev"
		;;
	*)
		error "Unexpected fstype for rootfs: $fstype. Must be ext4 or btrfs"
	esac
	mount_rootfs "$dev" "/target" || error "Could not mount newly created rootfs"

	mkdir -p /target/boot /target/mnt /target/target
	touch /target/.created
	unshare -m sh -c 'mount --bind /live/rootfs /mnt && cp -a /mnt/. /target' \
		|| error "Could not copy existing fs over"

	# adjust target partition for its new number
	# cf. mkswu/scripts/post_rootfs.sh .created section
	if [ -e /target/etc/fw_env.config ]; then
		sed -i -e "s/boot[01]/boot${target_ab}/" /target/etc/fw_env.config \
			|| error "Could not update target fw_env.config"
	fi
	sed -i -e "s/boot_[01]/boot_${target_ab}/" /target/etc/fstab \
		|| error "Could not update target fstab"
}

rollback_clone_copy_appfs() {
	# clone containers, cf. mkswu's pre_appfs
	info "Updating appfs snapshots"

	unshare -m sh -ec "cd /; abos-ctrl umount /mnt
		mount -t btrfs '$appdev' /mnt
		del_vol() {
			local vol
			for vol in \"\$@\"; do
				if [ -d \"\$vol\" ]; then
					btrfs -q subvolume delete \"\$vol\" \
						|| rm -rf \"\$vol\"
				fi
			done
		}
		del_vol '/mnt/boot_${target_ab}/containers_storage' \
			'/mnt/boot_${target_ab}/volumes'
		btrfs -q subvolume snapshot '/mnt/boot_${ab}/volumes' '/mnt/boot_${target_ab}/volumes'
		btrfs -q subvolume snapshot '/mnt/boot_${ab}/containers_storage' '/mnt/boot_${target_ab}/containers_storage'
		" || error "Could not delete containers storage subvolume"
}

rollback_clone_copy_boot() {
	# copy boot image, resetting env in the process with target's fw_env.config
	local source="${rootdev}boot${ab}" target="${rootdev}boot${target_ab}"
	local flash_dev="${target##*/}" env_sz env_offset
	# skip if no fw_env.config or boot partition
	if ! [ -e /etc/fw_env.config ] || ! [ -e "$target" ]; then
		return
	fi

	# from mkswu's scripts/post_boot.sh
	# Get environment device, offset and size
	# This assumes a single device contains env with nothing in
	# between redundant envs
	env_offset=$(awk '/^[^#]/ && $2 > 0 {
			if (!start || $2 < start)
				start = $2;
			if (!end || $2 + $3 > end)
				end = $2 + $3;
			if (dev && dev != $1) {
				print "Multiple devices in fw_env.config is not supported" > "/dev/stderr"
				exit(1)
			}
			dev=$1
		}
		END {
			if (!start) exit(1);
			printf("%d,%d\n", start, end-start);
		}
		' < /target/etc/fw_env.config) \
		|| error "Could not get boot env location"
	env_sz="${env_offset##*,}"
	env_offset="${env_offset%,*}"

	if cmp -s -n "$env_offset" "$source" "$target"; then
		info "Reusing up-to-date bootloader"
		return
	fi

	info "Cloning bootloader"

	if ! echo 0 > "/sys/block/$flash_dev/force_ro" \
	    || ! dd if="$source" of="$target" bs=1M count="$env_offset" \
			iflag=count_bytes conv=fsync status=none \
	    || ! dd if=/dev/zero of="$target" bs="$env_sz" count=1 \
			seek="$env_offset" oflag=seek_bytes \
			conv=fsync status=none; then
		echo 1 > "/sys/block/$flash_dev/force_ro"
		error "boot partition copy failed"
	fi
	echo 1 > "/sys/block/$flash_dev/force_ro"

	if stat /boot/uboot_env.d/* >/dev/null 2>&1; then
		cat /boot/uboot_env.d/* \
			| fw_setenv_nowarn --config /target/etc/fw_env.config \
				--script /dev/stdin --defenv /dev/null \
			|| error "Could not reset env for uboot copy"
	fi
}

ctrl_rollback_clone() {
	local appdev dev extlinux=""
	local cloned_rootfs="" enable_rollback=""
	local fstype=""

	while [ "$#" -gt 0 ]; do
		case "$1" in
		"--fstype")
			[ "$#" -ge 2 ] || error "$1 requires an argument";
			case "$2" in
			btrfs|ext4) fstype="$2";;
			*) error "$2 must be btrfs or ext4";;
			esac
			shift
			;;
		"-q"|"--quiet")
			debug=$((debug-1))
			;;
		"-v"|"--verbose")
			debug=$((debug+1))
			;;
		*)
			warning "Unrecognized argument $1"
			rollback_clone_help
			exit 1
			;;
		esac
		shift
	done

	get_appdev
	dev_from_current
	dev="$target_rootpart"

	# check we're free to flash first,
	# and take swupdate (mkswu) lock
	lock_update
	trap unlock_update EXIT INT QUIT TERM
	is_mountpoint /target \
		&& error "Something is already mounted in /target"

	info "Starting clone to $target_rootpart"

	# mark system unfit for rollback until we're done
	if fw_printenv_nowarn upgrade_available | grep -qxE 'upgrade_available=[012]'; then
		fw_setenv_nowarn upgrade_available 00 \
			|| error "Could not mark clone in progress"
	fi
	# only enable rollback if we normally would from uboot env:
	# read files alphabetically and keep track of last...
	if cat /boot/uboot_env.d/* | awk -F= '
			/^upgrade_available=/ { val=$2 }
			END { if (!val) exit(1); }'; then
		enable_rollback=1
	fi

	trap rollback_clone_cleanup EXIT INT QUIT TERM


	rollback_clone_copy_rootfs
	rollback_clone_copy_appfs
	rollback_clone_copy_boot

	if [ -n "$enable_rollback" ]; then
		# make other side able to rollback to current side
		fw_setenv_nowarn -c /target/etc/fw_env.config \
			upgrade_available 2
	fi

	[ -e /target/.created ] && rm -f /target/.created
	umount /target
	sync

	if [ -n "$enable_rollback" ]; then
		# ... and make this side able to rollback to other side
		fw_setenv_nowarn upgrade_available 2
	fi

	if [ -n "$cloned_rootfs" ]; then
		ctrl_atlog_write "rollback-clone" "Cloned system to ${dev#/dev/}"
		if [ -d /var/log/swupdate ] || mkdir /var/log/swupdate; then
			echo "$rootpart $(date +%s)" > /var/log/swupdate/last_update
			cp /etc/sw-versions "/var/log/swupdate/sw-versions-${target_rootpart#/dev/}"
		fi
	fi

	rollback_clone_cleanup

	info "Rollback clone successful"
}

