# SPDX-License-Identifier: MIT

# also uses helpers from podman_storage
. "$scripts_dir/podman_storage.sh"
. "$scripts_dir/btrfs.sh"

container_clear_help() {
	echo "Usage: abos-ctrl [common options] container-clear [extra options]"
	echo
	echo "  --force   do not prompt, and do not check if containers are running"
	echo "            (running containers are stopped forcefully)"
	echo
	echo "see abos-ctrl --help for common options"
}

ctrl_container_clear() {
	local appdev force=""
	get_appdev

	while [ "$#" -gt 0 ]; do
		case "$1" in
		"-f"|"--force")
			force=1
			;;
		"-q"|"--quiet")
			debug=$((debug-1))
			;;
		"-v"|"--verbose")
			debug=$((debug+1))
			;;
		-[vq]*)
			set -- "$@" "${1:0:2}" "-${1:2}"
			;;
		*)
			warning "Invalid option $1"
			container_clear_help >&2
			exit 1
			;;
		esac
		shift
	done

	if [ -z "$force" ]; then
		echo "This command will remove all containers and related data."
		echo " - The following file and directories will be removed:"
		echo "   - /var/app/rollback/volumes/"
		echo "   - /var/app/volumes/"
		echo "   - /etc/atmark/containers/*.conf"
		echo " - Lines containing the word \"container\" will be deleted from the following files:"
		echo "   - /etc/sw-versions"
		echo "   - /etc/swupdate.watch"
		echo "Continue? [y/N]"
		prompt_yesno n || error "Canceled."

		podman_check_running
	fi

	sed -i -e '/container/d' /etc/swupdate.watch \
		&& persist_file /etc/swupdate.watch \
		|| error "Could not modify /etc/swupdate.watch"
	sed -i -e '/container/d' /etc/sw-versions \
		&& persist_file /etc/sw-versions \
		|| error "Could not modify /etc/sw-versions"
	rm -f /etc/atmark/containers/*.conf \
		&& persist_file -r /etc/atmark/containers \
		|| error "Could not remove container configurations /etc/atmark/containers/*.conf"

	podman pod rm -a -f >/dev/null \
		|| error "Could not remove pod"
	podman rm -a -f >/dev/null \
		|| error "Cloud not remove container"

	trap mount_cleanup_basemount EXIT INT QUIT TERM

	basemount=$(mktemp -d btrfs-root.XXXXXX) \
		|| error "Could not create tmpdir"
	mount -t btrfs "$appdev" "$basemount" \
		|| error "Could not mount appfs"

	btrfs_subvol_delete "boot_0/containers_storage"
	btrfs_subvol_delete "boot_0/volumes"
	btrfs_subvol_delete "boot_1/containers_storage"
	btrfs_subvol_delete "boot_1/volumes"
	# A6E uses subvolumes for device-local data
	btrfs_subvol_recursive_delete "/volumes"
	if btrfs_subvol_delete "containers_storage"; then
		btrfs_subvol_create "containers_storage"
	fi

	umount_if_mountpoint /var/lib/containers/storage_readonly/overlay
	umount_if_mountpoint /var/lib/containers/storage_readonly
	umount_if_mountpoint /var/lib/containers/storage/overlay
	umount_if_mountpoint /var/lib/containers/storage
	umount_if_mountpoint /var/app/rollback/volumes
	umount_if_mountpoint /var/app/volumes

	btrfs_subvol_create "boot_0/containers_storage"
	btrfs_subvol_create "boot_0/volumes"
	btrfs_subvol_create "boot_1/containers_storage"
	btrfs_subvol_create "boot_1/volumes"
	btrfs_subvol_create "volumes"

	mount /var/lib/containers/storage_readonly \
		|| error "Could not mount containers_storage subvol"
	mount /var/app/rollback/volumes \
		|| error "Could not mount rollback/volume subvol"
	mount /var/app/volumes \
		|| error "Could not mount volume subvol"
	if [ -e "$basemount/containers_storage" ]; then
		mount /var/lib/containers/storage \
			|| error "Could not mount persistent storage"
	fi

	umount "$basemount" \
		|| error "Could not umount tmpdir $basemount"
	rmdir "$basemount" \
		|| error "Could not remove tmpdir $basemount"
	echo "Remove all container data succeeded"
}
