#!/bin/bash
# SPDX-License-Identifier: MIT

cd "$(dirname "$0")" || exit

error() {
	[ -n "$timespec" ] && echo -n "$base / $timespec: " >&2
	printf "%s\n" "$@" >&2
	exit 1
}

build_faketime() {
	[ -e faketime/libfaketime.so.1 ] && return
	(
		cd faketime || exit
		cc -o libfaketime.so.1 -Wl,-soname,libfaketime.so.1  \
			-Wl,--version-script=libfaketime.map -shared \
			-std=gnu99 -Wall -Wextra -Werror -DFAKE_PTHREAD \
			-DFAKE_STAT -DFAKE_UTIME -DFAKE_SLEEP -DFAKE_TIMERS \
			-DFAKE_INTERNAL_CALLS -fPIC -Wno-nonnull-compare \
			libfaketime.c -ldl -lm -lrt -lpthread
	) || exit
}

run_date() {
	# date -d can set an offset, but we don't want to mess with tests:
	# use libfaketime
	LD_PRELOAD=./faketime/libfaketime.so.1 \
		FAKETIME="$base" \
		date -d "$1" +%s
}

run_schedts() {
	../../tools/schedule_ts -B "$base" -- "$1"
}

compare_outputs() {
	local timespec="$1"
	local date schedts
	[ -n "$base" ] || base="$(date +"%Y-%m-%d %H:%M:%S")"

	date="$(run_date "$timespec" 2>/dev/null)" || error "date failed"
	schedts="$(run_schedts "$timespec")" || error "schedule_ts failed"

	[ "$date" = "$schedts" ] \
		|| error "Bad timestamp $schedts ($(date -d @"$schedts")) instead of $date ($(date -d @"$date"))"
}

assert_fail() {
	local timespec="$1"

	! run_schedts "$timespec" >/dev/null 2>&1 \
	       || error "schedule_ts didn't fail as it should"
}

compare_realtime() {
	# check time obtention works (without -B)
	local timespec="$1"

	local date schedts
	date=$(date -d "$timespec" +%s)
	schedts=$(../../tools/schedule_ts "$timespec")

	# hopefully won't take more than 10 seconds to run even with slow emulation...
	[ "$((date > schedts ? date - schedts : schedts - date))" -lt 10 ] \
		|| error "Bad timestamp $schedts ($(date -d @"$schedts")) instead of $date ($(date -d @"$date"))"
}

test_random_date() {
	local days=( tomorrow week day today
		monday tuesday thursday wednesday thursday friday )
	local day_modifiers=( '' last this next first third fourth eleventh )
	local meridians=( '' am p.m. )
	local pads=( %d %02d )
	local colons=( '' : )
	local time pad colon modifier

	colon="${colons[$((RANDOM%2))]}"
	case "$((RANDOM%3))" in
	0) # seconds, colon mandatory
		colon=:
		pad="${pads[$((RANDOM%2))]}"
		# shellcheck disable=SC2059 # variable for print format is intended
		printf -v time "$pad" "$((RANDOM%60))"
		;&
	1) # minutes
		pad="${pads[$((RANDOM%2))]}"
		printf -v time "$pad$colon%s" "$((RANDOM%60))" "$time"
		;&
	2) # hour
		pad="${pads[$((RANDOM%2))]}"
		printf -v time "$pad$colon%s" "$((RANDOM%24))" "$time"
		;;
	esac

	modifier=${meridians[$((RANDOM%${#meridians[@]}))]}
	[ -n "$modifier" ] && time="$time $modifier"
	modifier=${day_modifiers[$((RANDOM%${#day_modifiers[@]}))]}
	[ -n "$modifier" ] && time="$time $modifier"
	modifier=${days[$((RANDOM%${#days[@]}))]}
	[ -n "$modifier" ] && time="$time $modifier"

	local date schedts
	date="$(run_date "$time" 2>/dev/null)" || return 0 # skip invalid date
	schedts="$(run_schedts "$time")" || error "$time: schedule_ts failed"

	[ "$date" = "$schedts" ] \
		|| error "$time: Bad timestamp $schedts ($(date -d @"$schedts")) instead of $date ($(date -d @"$date"))"
}

compare_realtime now
compare_realtime tomorrow

build_faketime

# being sourced?
if [[ $(caller | cut -d' ' -f1) != "0" ]]; then
	error() { printf "ERROR: %s\n" "$@"; }
	return 0
fi

RANDOM_SEED=${RANDOM_SEED:-$RANDOM}
RANDOM=$RANDOM_SEED
echo "Running with RANDOM_SEED=$RANDOM_SEED"

for _ in {1..10}; do
	base="$(date -d "@$((RANDOM*65535+RANDOM))" +"%Y-%m-%d %H:%M:%S")"
	echo "base $(date -d "$base")"
	# basic tests
	compare_outputs "next week"
	compare_outputs "0 tomorrow"
	compare_outputs "0015 tomorrow"
	compare_outputs "003 am"
	compare_outputs "00:15 tomorrow"
	compare_outputs "0:15 tomorrow"
	compare_outputs "00:00:15 tomorrow"
	compare_outputs "01:15 tomorrow"
	compare_outputs "01:00:15 tomorrow"
	compare_outputs "1:15 tomorrow"
	compare_outputs "1:00:15 tomorrow"
	compare_outputs "01:0:15 tomorrow"
	compare_outputs "1:0:15 tomorrow"
	compare_outputs "00:01:15 tomorrow"
	compare_outputs "3 2 days"
	compare_outputs "5 pm next week"
	compare_outputs "705 2 weeks"
	compare_outputs "8:12 next month"
	compare_outputs "09:04:23 pm 2 months"
	compare_outputs "23 next monday"
	compare_outputs "4pm 2 monday"
	compare_outputs "2 min 3 secs"
	compare_outputs "- 3 hour"
	compare_outputs "now + 3 hour"
	compare_outputs "now + next sun"
	compare_outputs "now + 3 sec 3:00"
	compare_outputs "now + 3 sec next monday"
	compare_outputs "3 hour 2 year"
	compare_outputs "oct 3 3 2012"
	compare_outputs "3 oct 2012 1712"
	compare_outputs "3:00 pm"
	compare_outputs "12:00pm"
	compare_outputs "12am"

	# we don't really care about YYMMDD, but might as well..
	compare_outputs '000101' # year 2000
	compare_outputs '00101' # year 0
	compare_outputs '00000101' # year 0

	# all check days of the week
	for day in sunday mon tue wed thu friday sat; do
		compare_outputs "$day 3 3 min ago"
		compare_outputs "next $day"
		compare_outputs "3 $day 1"
		compare_outputs "3 last $day"
	done

	# fuzz a bit
	for _ in {0..100}; do
		test_random_date
	done
done

# check all months just once
for month in jan february march apr may june july aug sept oct nov dec; do
	compare_outputs "01 $month"
done

# check common failures fail
assert_fail "now + 3"
assert_fail "now + 3s"
assert_fail "now + 3 s"
assert_fail "- 3"

# known bugs:
# '120120 third thursday' (YYMMDD + weekday doesn't really make sense anyway...)
