#!/bin/bash

# shellcheck disable=SC2034 # unused variables, used in sourced functions

error() {
	printf "%s\n" "$@" >&2
	exit 1
}

cd "$(dirname "$0")" || error "cd test dir"

# option to regenerate output we compare with
# use with care if podman_start output changed
# or after confirming new tests work
REGENERATE="${REGENERATE:-}"

[ -z "$REGENERATE" ] || rm -f podman_start/*/output.*

# add tools to path for schedule_ts, used in some tests
PATH=$PWD/../tools:$PATH

check_output() {
	local output="$1"
	shift
	local cmd=( "$@" )

	echo "${cmd[*]}"

	if [ -n "$REGENERATE" ]; then
		"${cmd[@]}" > "$d/output.$output"
	else
		diff -u "$d/output.$output" <("${cmd[@]}") \
			|| error "'${cmd[*]}' produced different output than expected"
	fi
}
podman_start() {
	TEST_HOTPLUG=1 \
	../podman/podman_start \
		--networksdir "$TESTDIR/networks" \
		--statedir "$TESTDIR/state" \
		"$@" 2>&1
}


# prepare and clean test dir.
# we need a stable path for this as it appears in logs,
# so we can't use mktemp
rm -rf /tmp/podman_start
mkdir /tmp/podman_start || error "Need /tmp/podman_start to be writable"
export TESTDIR="/tmp/podman_start"

# verify dry run mode, always
for d in podman_start/*; do
	[ -d "$d" ] || continue
	check_output all podman_start \
		-n -c "$d" -a
	for f in "$d/"*.conf; do
		b=$(basename "$f" .conf)
		check_output "$b" podman_start -n -c "$d" "$b"
	done
done

# cleanup temp dir
rm -rf /tmp/podman_start

# if podman is available also run some containers
command -v podman > /dev/null || exit 0

# setup test requirements
TESTDIR=$(mktemp -d /tmp/abos-base.podman.XXXXXX) \
	|| error "Could not create temp dir"
# shellcheck disable=SC2064 # want to expand TESTDIR now...
trap "find '$TESTDIR' -xdev -exec sh -c 'for f in \"\$@\"; do
		abos-ctrl internal - is_mountpoint \"\$f\" && umount \"\$f\";
	done' - {} +; rm -rf '$TESTDIR'" EXIT
mkdir "$TESTDIR/test space"
echo foo > "$TESTDIR/test space/foo"
mkdir "$TESTDIR/test"
echo bar > "$TESTDIR/test/bar"
podman image exists docker.io/alpine \
	|| podman pull docker.io/alpine

for d in podman_start/run_*; do
	for f in "$d"/*.conf; do
		b=$(basename "$f" .conf)
		if podman container exists "$b"; then
			podman rm "$b" ||error "Could not cleanup $b, still running?"
		fi
		podman_start -v -c "$d" "$b" \
			|| error "podman_start -c $d $b failed"
		if grep -q set_healthcheck "$f"; then
			# We're testing all commands here, this unrolls as follow:
			# 1-3s: success
			# 4-5s: fail twice, podman_start restart
			# 6-8s: recovery/success
			# more fail if we wait longer, just kill it.
			# ... then wait a bit more to ensure no action was run when
			# container was killed
			i=0
			while ! [ -e "$TESTDIR/health_recovery" ]; do
				sleep 1
				((i++ > 10)) && error "healthcheck recovery didn't happen in 10s"
			done
			podman kill "$b" || error "could not kill $b ($d)"
			[ "$(wc -l < "$TESTDIR/health_start")" = 1 ] \
				|| error "start hook was not run exactly once"
			[ "$(wc -l < "$TESTDIR/health_fail")" = 1 ] \
				|| error "fail hook was not run exactly once"
			[ "$(wc -l < "$TESTDIR/health_recovery")" = 1 ] \
				|| error "recovery hook was not run exactly once"
		fi
		if ! timeout 10s podman wait "$b"; then
			podman kill "$b"
			error "container $b ($d) didn't stop within timeout?"
		fi
		check_output "run_$b" podman logs "$b"
		podman rm "$b"
	done
done

# check hook ran
[ "$(cat "$TESTDIR/state/hooks_one")" = "ran
ran" ] || error "first hook didn't run twice: $(cat "$TESTDIR/state/hooks_one")"
[ "$(cat "$TESTDIR/state/hooks_two")" = "ran" ] \
	|| error "second hook didn't run once: $(cat "$TESTDIR/state/hooks_two")"

# check healthcheck thread stopped
[ -e "$TESTDIR/state/healthcheck_healthcheck.pid" ] \
	&& error "healthcheck pid file still present"

# udevfw will still be running, but restarting the hotplug container
# should replace it...
# XXX replace with `pgrep -f udevfw && error` once stop hook fixed to handle restart
udevfw_pid=$(cat "$TESTDIR/state/hotplug-hooks.d/udevfw.pid") \
	|| error "no udevfw.pid"
grep -q udevfw "/proc/$udevfw_pid/cmdline" \
	|| error "udevfw pid $udevfw_pid isn't udevfw (or its dummy)"
podman_start -v -c podman_start/run_containers hotplug \
	|| error "starting hotplug container again failed"
timeout 10s podman wait hotplug \
	|| error "second hotplug container start didn't finish in 10s"
grep -q udevfw "/proc/$udevfw_pid/cmdline" 2>/dev/null \
	&& error "old udevfw pid $udevfw_pid is still running"
udevfw_pid=$(cat "$TESTDIR/state/hotplug-hooks.d/udevfw.pid") \
	|| error "no new udevfw.pid"
grep -q udevfw "/proc/$udevfw_pid/cmdline" \
	|| error "new udevfw pid $udevfw_pid isn't udevfw (or its dummy)"
kill -9 "$udevfw_pid"

# test podman network initialization.
# This has to run as root, on alpine -- and 'podman network inspect'
# does not work in a container... grmblgrmblgrmbl.
if grep -qx "ID=alpine" /etc/os-release && [ "$(id -u)" = 0 ]; then (
	. ../podman/podman_start --source

	NETWORKSDIR="$TESTDIR/networks"
	rm -rf "$NETWORKSDIR"

	init_network

	podman network inspect podman \
		| sed -e '1d;$d' \
		      -e 's/\(dns_enabled.*\)false/\1true/' \
		      -e 's/^ *//' -e 's/"created": "[^"]*"/"created": ""/' \
		      -e 's/"id": "[^"]*"/"id": ""/' \
		      > "$NETWORKSDIR/podman.json_podman_inspect"
	sed -e 's/^ *//' -e 's/"created": "[^"]*"/"created": ""/' \
	    -e 's/"id": "[^"]*"/"id": ""/' \
		< "$NETWORKSDIR/podman.json" \
		> "$NETWORKSDIR/podman.json_podman_start"

	diff -u "$NETWORKSDIR/podman.json_podman_inspect" \
		"$NETWORKSDIR/podman.json_podman_start" \
		|| error "generated podman network config differs from podman's"

) || error "init_network test failed"
fi

# test mklink_hostpath directly
if [ "$(id -u)" = 0 ]; then (
	. ../podman/podman_start --source
	STATEDIR="$TESTDIR/state"
	NETWORKSDIR="$TESTDIR/networks"
	name=contname
	devnum=0

	# dir
	hostpath="$TESTDIR/test space"
	mklink_hostpath
	[ "$hostpath" = "$STATEDIR/${name}_0" ] \
		|| error "hostpath $hostpath not correct (dir)"
	[ -e "$hostpath"/foo ] || error "hostpath not mounted"
	podman run -v "$hostpath:/work" --rm docker.io/alpine test -e /work/foo \
		|| error "podman did not handle mount correctly (dir)"

	# file
	hostpath="$TESTDIR/test space/foo"
	mklink_hostpath
	[ "$hostpath" = "$STATEDIR/${name}_1" ] \
		|| error "hostpath $hostpath not correct (file)"
	[ "$(cat "$hostpath")" = "foo" ] || error "hostpath content not correct"
	[ "$(podman run -v "$hostpath:/work" --rm docker.io/alpine cat /work)" = "foo" ] \
		|| error "podman did not handle mount correctly (file)"

	# remove dir/file
	devnum=0
	hostpath="$TESTDIR/test space/foo"
	mklink_hostpath
	[ "$hostpath" = "$STATEDIR/${name}_0" ] \
		|| error "hostpath $hostpath not correct (remove dir)"
	hostpath="$TESTDIR/test space/foo"
	mklink_hostpath
	[ "$hostpath" = "$STATEDIR/${name}_1" ] \
		|| error "hostpath $hostpath not correct (remove file)"
) || error "mklist_hostpath test failed"
fi

if command -v udevfw >/dev/null && [ "$(id -u)" = 0 ]; then (
	# really test hotplug
	unset TEST_HOTPLUG
	unset podman_start
	podman_start -v -c podman_start/hotplug watch \
		|| error "hotplug watch container didn't start"
	mkfifo "$TESTDIR/watch.pipe"
	podman logs -f watch > "$TESTDIR/watch.pipe" &
	watch_log_pid="$!"
	timeout 10s grep -q -m 1 "watch setup" "$TESTDIR/watch.pipe" \
		|| error "watch container didn't set up watch"
	podman_start -v -c podman_start/hotplug uinput \
		|| error "uinput container didn't start"
	timeout 10s grep -q -m 1 "added input" "$TESTDIR/watch.pipe" \
		|| error "watch container didn't pick up new input"
	kill "$watch_log_pid" 2>/dev/null
) || error "hotplug tests failed"
fi
